// 内容脚本：鼠标悬停图片识别功能

class ImageHoverDetector {
    constructor() {
        this.encryptedApiKey = 'Nzk1MDE3NmEtZTg2MC00MzU2LWE1NzktNWM1N2JhNjM0N2Jl';
        this.apiKey = this.decryptApiKey(this.encryptedApiKey);
        this.modelId = 'ep-20250801193632-gcklq';
        this.apiUrl = 'https://ark.cn-beijing.volces.com/api/v3/chat/completions';
        this.initEventListeners();
    }

    decryptApiKey(encryptedKey) {
        return atob(encryptedKey);
    }

    initEventListeners() {
        // 监听文档中所有图片的鼠标悬停事件
        document.addEventListener('mouseover', (e) => {
            const img = e.target.closest('img');
            if (img && !img.dataset.hoverDetector) {
                this.setupImageHover(img);
            }
        }, true);
    }

    setupImageHover(img) {
        // 标记图片已处理
        img.dataset.hoverDetector = 'true';
        img.style.cursor = 'pointer';

        // 添加悬停事件监听器
        img.addEventListener('mouseenter', () => {
            this.currentImage = img;
            this.createHoverTooltip(img);
        });

        // 添加点击事件监听器（可选，用于触发识别）
        img.addEventListener('click', () => {
            this.analyzeHoveredImage();
        });
    }

    createHoverTooltip(img) {
        // 移除已存在的提示框
        this.removeTooltip();

        // 创建悬停提示框
        this.tooltip = document.createElement('div');
        this.tooltip.className = 'image-detector-tooltip';
        this.tooltip.style.cssText = `
            position: absolute;
            padding: 8px 12px;
            background-color: rgba(0, 0, 0, 0.8);
            color: white;
            border-radius: 4px;
            font-size: 14px;
            z-index: 10000;
            pointer-events: none;
            transition: opacity 0.2s;
        `;
        this.tooltip.textContent = '点击识别图片内容';

        // 计算提示框位置
        const rect = img.getBoundingClientRect();
        const scrollX = window.scrollX || document.documentElement.scrollLeft;
        const scrollY = window.scrollY || document.documentElement.scrollTop;

        // 显示在图片上方居中位置
        this.tooltip.style.left = `${rect.left + scrollX + rect.width / 2 - this.tooltip.offsetWidth / 2}px`;
        this.tooltip.style.top = `${rect.top + scrollY - 30}px`;

        document.body.appendChild(this.tooltip);
    }

    removeTooltip() {
        if (this.tooltip && this.tooltip.parentNode) {
            this.tooltip.parentNode.removeChild(this.tooltip);
            this.tooltip = null;
        }
    }

    async analyzeHoveredImage() {
        if (!this.currentImage) return;

        // 移除提示框
        this.removeTooltip();

        // 创建加载弹窗
        this.createLoadingModal();

        try {
            // 获取图片数据
            const imageData = await this.getImageData(this.currentImage);
            // 调用AI识别
            const response = await this.callAI(imageData);
            // 显示结果
            this.displayResults(response);
        } catch (error) {
            console.error('Analysis error:', error);
            this.showError('识别失败，请重试');
        }
    }

    getImageData(img) {
        return new Promise((resolve, reject) => {
            // 如果图片已经加载完成
            if (img.complete) {
                const canvas = document.createElement('canvas');
                const ctx = canvas.getContext('2d');
                if (!ctx) {
                    reject(new Error('无法创建Canvas上下文'));
                    return;
                }

                canvas.width = img.width;
                canvas.height = img.height;
                ctx.drawImage(img, 0, 0);

                try {
                    const dataUrl = canvas.toDataURL('image/jpeg');
                    resolve(dataUrl);
                } catch (error) {
                    // 如果由于安全原因无法转换，尝试直接使用图片URL
                    if (img.src.startsWith('http') || img.src.startsWith('data:image')) {
                        resolve(img.src);
                    } else {
                        reject(new Error('无法获取图片数据'));
                    }
                }
            } else {
                // 图片未加载完成，等待加载完成
                img.addEventListener('load', () => {
                    this.getImageData(img).then(resolve).catch(reject);
                });
                img.addEventListener('error', () => {
                    reject(new Error('图片加载失败'));
                });
            }
        });
    }

    async callAI(imageData) {
        const payload = {
            model: this.modelId,
            messages: [{
                role: 'user',
                content: [{
                    type: 'text',
                    text: '请识别这张图片中的所有物体，并统计每种物体的数量。对于动植物、花鸟虫鱼等生物类物体，请提供其学名、体表特征、习性、分布区域等详细信息。同时提供图片的详细描述，包括场景、物体位置关系、颜色等信息。请以JSON格式返回结果，格式为：{"description": "图片详细描述", "objects": [{"name": "物体名称", "scientificName": "学名", "physicalFeatures": "体表特征", "habitat": "习性", "distribution": "分布区域", "count": 数量}]}。只返回JSON，不要其他文字。'
                }, {
                    type: 'image_url',
                    image_url: {
                        url: imageData
                    }
                }]
            }]
        };

        const response = await fetch(this.apiUrl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${this.apiKey}`
            },
            body: JSON.stringify(payload)
        });

        if (!response.ok) {
            throw new Error(`API request failed: ${response.status}`);
        }

        const data = await response.json();
        return this.parseAIResponse(data);
    }

    parseAIResponse(data) {
        try {
            const content = data.choices[0].message.content;
            const jsonMatch = content.match(/\{[\s\S]*\}/);
            if (jsonMatch) {
                return JSON.parse(jsonMatch[0]);
            }
            throw new Error('No valid JSON found in response');
        } catch (error) {
            console.error('Parse error:', error);
            return { objects: [{ name: '识别结果', count: '解析失败' }] };
        }
    }

    createLoadingModal() {
        // 移除已存在的弹窗
        this.removeModal();

        // 创建半透明背景
        this.modalOverlay = document.createElement('div');
        this.modalOverlay.className = 'image-detector-overlay';
        this.modalOverlay.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 9999;
            display: flex;
            align-items: center;
            justify-content: center;
        `;

        // 创建加载弹窗
        this.modal = document.createElement('div');
        this.modal.className = 'image-detector-modal';
        this.modal.style.cssText = `
            background-color: white;
            border-radius: 8px;
            padding: 20px;
            width: 400px;
            max-width: 90%;
            text-align: center;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        `;

        // 添加加载动画
        const spinner = document.createElement('div');
        spinner.style.cssText = `
            width: 40px;
            height: 40px;
            border: 4px solid #f3f3f3;
            border-top: 4px solid #007bff;
            border-radius: 50%;
            margin: 0 auto 15px;
            animation: spin 1s linear infinite;
        `;

        // 添加动画样式
        const style = document.createElement('style');
        style.textContent = `
            @keyframes spin {
                0% { transform: rotate(0deg); }
                100% { transform: rotate(360deg); }
            }
        `;
        document.head.appendChild(style);

        // 添加加载文本
        const text = document.createElement('p');
        text.textContent = '正在识别图片内容...';
        text.style.margin = '0';
        text.style.color = '#333';
        text.style.fontSize = '14px';

        // 组装弹窗
        this.modal.appendChild(spinner);
        this.modal.appendChild(text);
        this.modalOverlay.appendChild(this.modal);
        document.body.appendChild(this.modalOverlay);
    }

    displayResults(data) {
        // 移除加载弹窗
        this.removeModal();

        // 创建结果弹窗
        this.modalOverlay = document.createElement('div');
        this.modalOverlay.className = 'image-detector-overlay';
        this.modalOverlay.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 9999;
            display: flex;
            align-items: center;
            justify-content: center;
        `;

        // 点击背景关闭弹窗
        this.modalOverlay.addEventListener('click', (e) => {
            if (e.target === this.modalOverlay) {
                this.removeModal();
            }
        });

        // 创建结果弹窗
        this.modal = document.createElement('div');
        this.modal.className = 'image-detector-modal';
        this.modal.style.cssText = `
            background-color: white;
            border-radius: 8px;
            padding: 20px;
            width: 350px;
            max-width: 90%;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        `;

        // 添加标题
        const title = document.createElement('h3');
        title.textContent = '图片识别结果';
        title.style.margin = '0 0 15px 0';
        title.style.color = '#333';
        title.style.fontSize = '14px';
        title.style.textAlign = 'center';

        // 创建内容容器
        const contentDiv = document.createElement('div');
        contentDiv.style.marginBottom = '15px';

        // 显示图片描述
        if (data.description) {
            const description = document.createElement('div');
            description.style.marginBottom = '15px';
            description.style.padding = '10px';
            description.style.backgroundColor = '#f8f9fa';
            description.style.borderRadius = '6px';
            description.style.fontSize = '14px';
            description.style.lineHeight = '1.5';

            const descTitle = document.createElement('p');
            descTitle.innerHTML = '<strong>图片描述:</strong>';
            descTitle.style.margin = '0 0 5px 0';
            descTitle.style.color = '#333';

            const descText = document.createElement('p');
            descText.textContent = data.description;
            descText.style.margin = '0';
            descText.style.color = '#333';

            description.appendChild(descTitle);
            description.appendChild(descText);
            contentDiv.appendChild(description);
        }

        // 显示AI图片检测结果
        if (data.isAIImage !== undefined) {
            const aiDetectionDiv = document.createElement('div');
            aiDetectionDiv.style.marginBottom = '15px';
            aiDetectionDiv.style.padding = '10px';
            aiDetectionDiv.style.backgroundColor = '#f8f9fa';
            aiDetectionDiv.style.borderRadius = '6px';
            aiDetectionDiv.style.fontSize = '14px';
            
            const aiTitle = document.createElement('p');
            aiTitle.innerHTML = '<strong>AI图片检测:</strong>';
            aiTitle.style.margin = '0 0 5px 0';
            aiTitle.style.color = '#333';
            
            const aiResult = document.createElement('p');
            aiResult.textContent = `是否为AI生成: ${data.isAIImage ? '是' : '否'} (检测率: ${data.aiDetectionRate}%)`;
            aiResult.style.margin = '0';
            aiResult.style.color = '#333';
            
            aiDetectionDiv.appendChild(aiTitle);
            aiDetectionDiv.appendChild(aiResult);
            contentDiv.appendChild(aiDetectionDiv);
        }

        // 显示物体识别结果
        if (data.objects && data.objects.length > 0) {
            const objectsTitle = document.createElement('h4');
            objectsTitle.textContent = '识别到的物体:';
            objectsTitle.style.margin = '0 0 10px 0';
            objectsTitle.style.color = '#333';
            objectsTitle.style.fontSize = '14px';

            contentDiv.appendChild(objectsTitle);

            data.objects.forEach(obj => {
                const item = document.createElement('div');
                item.style.padding = '8px 0';
                item.style.borderBottom = '1px solid #eee';

                // 创建物体名称和数量的容器
                const nameCountContainer = document.createElement('div');
                nameCountContainer.style.display = 'flex';
                nameCountContainer.style.justifyContent = 'space-between';
                nameCountContainer.style.alignItems = 'center';

                // 物体名称
                const nameSpan = document.createElement('span');
                nameSpan.textContent = obj.name;
                nameSpan.style.fontWeight = '500';
                nameSpan.style.color = '#333';
                nameSpan.style.fontSize = '12px';

                // 学名
                if (obj.scientificName) {
                    const scientificNameSpan = document.createElement('span');
                    scientificNameSpan.textContent = ` (${obj.scientificName})`;
                    scientificNameSpan.style.fontStyle = 'italic';
                    scientificNameSpan.style.color = '#666';
                    scientificNameSpan.style.fontSize = '11px';
                    nameSpan.appendChild(scientificNameSpan);
                }

                // 数量
                const countSpan = document.createElement('span');
                countSpan.textContent = obj.count;
                countSpan.style.backgroundColor = '#007bff';
                countSpan.style.color = 'white';
                countSpan.style.padding = '2px 8px';
                countSpan.style.borderRadius = '12px';
                countSpan.style.fontSize = '12px';

                nameCountContainer.appendChild(nameSpan);
                nameCountContainer.appendChild(countSpan);
                item.appendChild(nameCountContainer);

                // 动植物详细信息 - 仅对生物类物体显示
                if (obj.scientificName && (obj.physicalFeatures || obj.habitat || obj.distribution)) {
                    const detailsDiv = document.createElement('div');
                    detailsDiv.style.marginTop = '5px';
                    detailsDiv.style.paddingTop = '5px';
                    detailsDiv.style.borderTop = '1px dashed #eee';
                    detailsDiv.style.fontSize = '11px';

                    if (obj.habitat) {
                        if (obj.physicalFeatures) {
                            const physicalFeaturesP = document.createElement('p');
                            physicalFeaturesP.innerHTML = `<strong>体表特征:</strong> ${obj.physicalFeatures}`;
                            physicalFeaturesP.style.margin = '2px 0';
                            physicalFeaturesP.style.color = '#555';
                            detailsDiv.appendChild(physicalFeaturesP);
                        }
                        if (obj.habitat) {
                            const habitatP = document.createElement('p');
                            habitatP.innerHTML = `<strong>习性:</strong> ${obj.habitat}`;
                            habitatP.style.margin = '2px 0';
                            habitatP.style.color = '#555';
                            detailsDiv.appendChild(habitatP);
                        }
                    }

                    if (obj.distribution) {
                        const distributionP = document.createElement('p');
                        distributionP.innerHTML = `<strong>分布:</strong> ${obj.distribution}`;
                        distributionP.style.margin = '2px 0';
                        distributionP.style.color = '#555';
                        detailsDiv.appendChild(distributionP);
                    }

                    item.appendChild(detailsDiv);
                }

                contentDiv.appendChild(item);
            });
        } else {
            const noResult = document.createElement('p');
            noResult.textContent = '未识别到物体';
            noResult.style.color = '#666';
            contentDiv.appendChild(noResult);
        }

        // 添加关闭按钮
        const closeBtn = document.createElement('button');
        closeBtn.textContent = '关闭';
        closeBtn.style.backgroundColor = '#007bff';
        closeBtn.style.color = 'white';
        closeBtn.style.border = 'none';
        closeBtn.style.padding = '10px 15px';
        closeBtn.style.borderRadius = '6px';
        closeBtn.style.cursor = 'pointer';
        closeBtn.style.fontSize = '14px';
        closeBtn.style.width = '100%';
        closeBtn.style.transition = 'background-color 0.3s';

        closeBtn.addEventListener('mouseover', () => {
            closeBtn.style.backgroundColor = '#0056b3';
        });

        closeBtn.addEventListener('click', () => {
            this.removeModal();
        });

        // 组装弹窗
        this.modal.appendChild(title);
        this.modal.appendChild(contentDiv);
        this.modal.appendChild(closeBtn);
        this.modalOverlay.appendChild(this.modal);
        document.body.appendChild(this.modalOverlay);
    }

    showError(message) {
        // 移除加载弹窗
        this.removeModal();

        // 创建错误弹窗
        this.modalOverlay = document.createElement('div');
        this.modalOverlay.className = 'image-detector-overlay';
        this.modalOverlay.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 9999;
            display: flex;
            align-items: center;
            justify-content: center;
        `;

        // 点击背景关闭弹窗
        this.modalOverlay.addEventListener('click', (e) => {
            if (e.target === this.modalOverlay) {
                this.removeModal();
            }
        });

        // 创建错误弹窗
        this.modal = document.createElement('div');
        this.modal.className = 'image-detector-modal';
        this.modal.style.cssText = `
            background-color: white;
            border-radius: 8px;
            padding: 20px;
            width: 400px;
            max-width: 90%;
            text-align: center;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        `;

        // 添加错误图标
        const errorIcon = document.createElement('div');
        errorIcon.textContent = '❌';
        errorIcon.style.fontSize = '48px';
        errorIcon.style.marginBottom = '15px';

        // 添加错误文本
        const errorText = document.createElement('p');
        errorText.textContent = message;
        errorText.style.margin = '0 0 15px 0';
        errorText.style.color = '#333';
        errorText.style.fontSize = '14px';

        // 添加关闭按钮
        const closeBtn = document.createElement('button');
        closeBtn.textContent = '确定';
        closeBtn.style.backgroundColor = '#007bff';
        closeBtn.style.color = 'white';
        closeBtn.style.border = 'none';
        closeBtn.style.padding = '10px 15px';
        closeBtn.style.borderRadius = '6px';
        closeBtn.style.cursor = 'pointer';
        closeBtn.style.fontSize = '14px';
        closeBtn.style.width = '100%';
        closeBtn.style.transition = 'background-color 0.3s';

        closeBtn.addEventListener('mouseover', () => {
            closeBtn.style.backgroundColor = '#0056b3';
        });

        closeBtn.addEventListener('click', () => {
            this.removeModal();
        });

        // 组装弹窗
        this.modal.appendChild(errorIcon);
        this.modal.appendChild(errorText);
        this.modal.appendChild(closeBtn);
        this.modalOverlay.appendChild(this.modal);
        document.body.appendChild(this.modalOverlay);
    }

    removeModal() {
        if (this.modalOverlay && this.modalOverlay.parentNode) {
            this.modalOverlay.parentNode.removeChild(this.modalOverlay);
            this.modalOverlay = null;
            this.modal = null;
        }
    }
}

// 页面加载完成后初始化
window.addEventListener('load', () => {
    const detector = new ImageHoverDetector();
    detector.initClipboardListener();
});

// 添加一个处理系统剪切板的辅助函数（兼容Windows和苹果系统）
ImageHoverDetector.prototype.initClipboardListener = function() {
    // 当用户在页面上按下粘贴快捷键时检查剪贴板（Windows: Ctrl+V, macOS: Command+V）
    document.addEventListener('paste', async (e) => {
        try {
            // 获取剪贴板数据
            if (e.clipboardData && e.clipboardData.items) {
                const items = e.clipboardData.items;
                
                for (let i = 0; i < items.length; i++) {
                    // 检查是否包含图片数据
                    if (items[i].type.indexOf('image') !== -1) {
                        // 创建加载弹窗
                        this.createLoadingModal();
                        
                        try {
                            const blob = items[i].getAsFile();
                            const imageData = await this.convertBlobToDataUrl(blob);
                            const response = await this.callAI(imageData);
                            this.displayResults(response);
                        } catch (error) {
                            console.error('Clipboard image analysis error:', error);
                            this.showError('剪切板图片识别失败，请重试');
                        }
                        
                        break;
                    }
                }
            }
        } catch (error) {
            console.error('Clipboard access error:', error);
            // 在苹果系统上可能会出现权限问题，可以尝试使用现代Clipboard API作为备选
            this.tryModernClipboardAPI();
        }
    });
};

// 尝试使用现代Clipboard API作为备选（更好的macOS兼容性）
ImageHoverDetector.prototype.tryModernClipboardAPI = async function() {
    try {
        // 检查浏览器是否支持现代Clipboard API
        if (navigator.clipboard && window.isSecureContext) {
            // 请求权限
            const permission = await navigator.permissions.query({ name: 'clipboard-read' });
            
            if (permission.state === 'granted' || permission.state === 'prompt') {
                // 尝试读取剪贴板中的图片
                const clipboardItems = await navigator.clipboard.read();
                
                for (const item of clipboardItems) {
                    // 检查是否包含图片
                    const types = item.types;
                    for (const type of types) {
                        if (type.startsWith('image/')) {
                            // 创建加载弹窗
                            this.createLoadingModal();
                            
                            try {
                                const blob = await item.getType(type);
                                const imageData = await this.convertBlobToDataUrl(blob);
                                const response = await this.callAI(imageData);
                                this.displayResults(response);
                            } catch (error) {
                                console.error('Modern Clipboard API error:', error);
                                this.showError('剪切板图片识别失败，请重试');
                            }
                            
                            return;
                        }
                    }
                }
            }
        }
    } catch (error) {
        console.error('Modern Clipboard API fallback failed:', error);
    }
};

// 将Blob转换为DataURL的辅助方法
ImageHoverDetector.prototype.convertBlobToDataUrl = function(blob) {
    return new Promise((resolve, reject) => {
        const reader = new FileReader();
        reader.onloadend = () => resolve(reader.result);
        reader.onerror = reject;
        reader.readAsDataURL(blob);
    });
};